from flask_shop import db
from werkzeug.security import generate_password_hash,check_password_hash #用于加密
from datetime import datetime  #注意这句引入

# 模型建立与映射
class BaseModel:
    create_time = db.Column(db.DateTime,default=datetime.now)
    update_time = db.Column(db.DateTime,default=datetime.now,onupdate=datetime.now)

class User(db.Model,BaseModel):
    __tablename__ = 't_user'
    id = db.Column(db.Integer,primary_key=True)
    name = db.Column(db.String(32),unique=True,nullable=False)
    pwd = db.Column(db.String(128))
    nick_name = db.Column(db.String(32))
    phone = db.Column(db.String(11))
    email = db.Column(db.String(20))

    rid = db.Column(db.Integer,db.ForeignKey('t_role.id'))
    role = db.relationship('Role',backref='users')
    
    # 模型完善，密码加密
    @property
    def password(self):   #此处的方法名password要与@password.setter的pasword同名
        return self.pwd
    
    @password.setter
    def password(self,t_pwd): #以后访问获取pwd或者给pwd重新赋值的时候，要用对象名.此处的方法名password
        self.pwd = generate_password_hash(t_pwd)  #密码加密
    
    def check_password(self,t_pwd):
        return check_password_hash(self.pwd,t_pwd)  #比较前台提交的密码和后台加密的密码,return不可少

    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'nick_name': self.nick_name,
            'phone': self.phone,
            'email': self.email,
            'role_name': self.role.name if self.role else ''
        }

# 定义第三张表关联角色与权限,实现多对多的映射
trm = db.Table('t_role_menu',
    db.Column('rid',db.Integer,db.ForeignKey('t_role.id'),primary_key=True),
    db.Column('mid',db.Integer,db.ForeignKey('t_menu.id'),primary_key=True)
)

# 菜单model的创建
class Menu(db.Model):
    __tablename__ = 't_menu'
    id = db.Column(db.Integer,primary_key=True)
    name = db.Column(db.String(32),unique=True,nullable=False)
    level = db.Column(db.Integer)
    path = db.Column(db.String(32))

    pid = db.Column(db.Integer,db.ForeignKey('t_menu.id')) # 这句代码执行时会报错？？？注意数据
    children = db.relationship('Menu')  #子菜单
    roles = db.relationship('Role',secondary=trm)

    # 对请求给出响应
    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'level': self.level, # 等级
            'path': self.path,
            'pid': self.pid   # 父节点
            # 'children': self.get_child_list() # 遍历子节点
        }
    
    # 遍历子节点
    def get_child_list(self):
        obj_child = self.children #获取子节点
        data = []
        for o in obj_child:
            data.append(o.to_dict())
        return data

class Role(db.Model):
    __tablename__ = 't_role'
    id = db.Column(db.Integer,primary_key=True)
    name = db.Column(db.String(32),unique=True,nullable=False)
    desc = db.Column(db.String(64))

    # users = db.relationship('User',backref='role') #可以这样写，但是更常见
    menus = db.relationship('Menu',secondary=trm)

    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'desc': self.desc,
            'menu': self.get_menu_dict()
        }
    
    # 角色接口携带权限数据
    def get_menu_dict(self):
        menu_list = []
        self.menus = sorted(self.menus,key=lambda mu:mu.id) # 角色的权限排序，这行代码表示根据对象的id排序，默认升序
        for m in self.menus:
            if m.level == 1:
                first_dict = m.to_dict()
                first_dict['children'] = []
                for s in self.menus:
                    if s.level == 2 and s.pid == m.id:
                        first_dict['children'].append(s.to_dict())
                menu_list.append(first_dict)
        return menu_list

# 商品分类model编写
class Category(db.Model):
    __tablename__ = 't_category'
    id = db.Column(db.Integer,primary_key=True)
    name = db.Column(db.String(32),nullable=False)
    level = db.Column(db.Integer)
    pid = db.Column(db.Integer,db.ForeignKey('t_category.id'))

    children = db.relationship('Category')

    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'level': self.level,
            'pid': self.pid
        }

# 分类参数Model创建
class Attribute(db.Model):
    __tablename__ = 't_attribute'
    id = db.Column(db.Integer,primary_key=True)
    name = db.Column(db.String(32))
    val = db.Column(db.String(255))
    cid = db.Column(db.Integer,db.ForeignKey('t_category.id'))
    _type = db.Column(db.Enum('static','dynamic'))

    category = db.relationship('Category',backref='attrs')

    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'val': self.val,
            'cid': self.cid,
            '_type': self._type
        }

# 商品管理Model的创建
class Goods(db.Model):
    __tablename__ = 't_goods'
    id = db.Column(db.Integer,primary_key=True)
    name = db.Column(db.String(512))
    price = db.Column(db.Float)  # 注意这里使用db.DECIMAI(20,4),在获取商品列表时会报错 TypeError: Object of type Decimal is not JSON serializable
    number = db.Column(db.Integer)
    introduce = db.Column(db.Text)
    big_log = db.Column(db.String(256))
    small_log = db.Column(db.String(256))
    state = db.Column(db.Integer) # 0 未通过 1 审核中 2 已审核
    is_promote = db.Column(db.Integer)
    hot_number = db.Column(db.Integer)
    weight = db.Column(db.Integer)
    cid_one = db.Column(db.Integer,db.ForeignKey('t_category.id'))
    cid_two = db.Column(db.Integer,db.ForeignKey('t_category.id'))
    cid_three = db.Column(db.Integer,db.ForeignKey('t_category.id'))

    category = db.relationship('Category',foreign_keys=[cid_three])

    def to_dict(self):
        return {
           'id':self.id,
            'name':self.name,
            'price':self.price,
            'number':self.number,
            'introduce':self.introduce,
            'big_log':self.big_log,
            'small_log':self.small_log,
            'state':self.state,
            'is_promote':self.is_promote,
            'hot_number':self.hot_number,
            'weight':self.weight,
            'cid_one':self.cid_one,
            'cid_two':self.cid_two,
            'cid_three':self.cid_three,
            'attrs':[a.to_dict() for a in self.category.attrs]
        }